//	TorusGames3DTiling.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt


#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"


void Make3DGameCellIntoTiling(
	double			m[4][4],
	signed int		x,			//	(x,y,z) is the center of the desired cell.
	signed int		y,
	signed int		z,
	TopologyType	aTopology)	//	aTopology defines the group of covering transformations.
{
	//	Construct a matrix m for the group element that
	//	maps the game cell to one of its translates in the tiling.

	//	Start with a pure translation to (x,y,z) and then...
	m[0][0] = 1.0;	m[0][1] = 0.0;	m[0][2] = 0.0;	m[0][3] = 0.0;
	m[1][0] = 0.0;	m[1][1] = 1.0;	m[1][2] = 0.0;	m[1][3] = 0.0;
	m[2][0] = 0.0;	m[2][1] = 0.0;	m[2][2] = 1.0;	m[2][3] = 0.0;
	m[3][0] =  x;	m[3][1] =  y;	m[3][2] =  z;	m[3][3] = 1.0;

	//	...account for the topology.
	switch (aTopology)
	{
		case Topology3DTorus:
			//	Pure translation is perfect.
			//	No adjustment needed.
			break;
		
		case Topology3DKlein:
			//	Cells at odd height y get flipped side-to-side (x ↔ -x).
			if ( ((unsigned int)(signed int)m[3][1]) & 0x00000001 )
				m[0][0] = -1.0;
			break;
		
		case Topology3DQuarterTurn:
			//	Cells get rotated a quarter turn counterclockwise
			//	(+x axis rotating to +y axis) for each unit
			//	of depth in the +z direction.
			switch ( ((unsigned int)(signed int)m[3][2]) & 0x00000003 )
			{
				case 0x00000000:
					//	No rotation
					break;

				case 0x00000001:
					//	1/4 CCW turn
					m[0][0] =  0.0;  m[0][1] =  1.0;
					m[1][0] = -1.0;  m[1][1] =  0.0;
					break;

				case 0x00000002:
					//	1/2 turn
					m[0][0] = -1.0;
					m[1][1] = -1.0;
					break;

				case 0x00000003:
					//	3/4 CCW turn
					m[0][0] =  0.0;  m[0][1] = -1.0;
					m[1][0] =  1.0;  m[1][1] =  0.0;
					break;
			}
			break;
		
		case Topology3DHalfTurn:
			//	Cells at odd depth z get rotated a half turn in the xy plane.
			if ( ((unsigned int)(signed int)m[3][2]) & 0x00000001 )
			{
				m[0][0] = -1.0;
				m[1][1] = -1.0;
			}
			break;
		
		default:
			GEOMETRY_GAMES_ABORT("Invalid TopologyType");
			break;
	}
}
